<?php
class User {
    // Connexion à la base de données et nom de la table
    private $conn;
    private $table_name = "users";
    
    // Propriétés de l'objet
    public $id;
    public $firstname;
    public $lastname;
    public $email;
    public $password;
    public $profile_image;
    public $is_online;
    public $last_login;
    public $created_at;
    
    // Constructeur avec $db comme connexion à la base de données
    public function __construct($db) {
        $this->conn = $db;
    }
    
    // Créer un nouvel utilisateur
    public function create() {
        $query = "INSERT INTO " . $this->table_name . "
                  SET firstname = :firstname,
                      lastname = :lastname,
                      email = :email,
                      password = :password,
                      is_online = 0,
                      created_at = NOW()";
        
        $stmt = $this->conn->prepare($query);
        
        // Nettoyer les données
        $this->firstname = htmlspecialchars(strip_tags($this->firstname));
        $this->lastname = htmlspecialchars(strip_tags($this->lastname));
        $this->email = htmlspecialchars(strip_tags($this->email));
        
        // Lier les valeurs
        $stmt->bindParam(":firstname", $this->firstname);
        $stmt->bindParam(":lastname", $this->lastname);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":password", $this->password);
        
        if ($stmt->execute()) {
            return true;
        }
        return false;
    }
    
    // Vérifier si l'email existe
    public function emailExists($email) {
        $query = "SELECT id, firstname, lastname, password
                  FROM " . $this->table_name . "
                  WHERE email = ?
                  LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $email);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            return true;
        }
        return false;
    }
    
    // Récupérer les informations d'un utilisateur par email
    public function getUserByEmail($email) {
        $query = "SELECT id, firstname, lastname, email, profile_image
                  FROM " . $this->table_name . "
                  WHERE email = ?
                  LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $email);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }
        return false;
    }
    
    // Connexion de l'utilisateur
    public function login($email, $password) {
        $query = "SELECT id, firstname, lastname, password
                  FROM " . $this->table_name . "
                  WHERE email = ?
                  LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $email);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (password_verify($password, $row['password'])) {
                $this->id = $row['id'];
                $this->firstname = $row['firstname'];
                $this->lastname = $row['lastname'];
                return true;
            }
        }
        return false;
    }
    
    // Lire les informations d'un utilisateur
    public function readOne() {
        $query = "SELECT u.id, u.firstname, u.lastname, u.email, u.profile_image, 
                         u.is_online, u.last_login, u.created_at
                  FROM " . $this->table_name . " u
                  WHERE u.id = ?
                  LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $this->firstname = $row['firstname'];
        $this->lastname = $row['lastname'];
        $this->email = $row['email'];
        $this->profile_image = $row['profile_image'];
        $this->is_online = $row['is_online'];
        $this->last_login = $row['last_login'];
        $this->created_at = $row['created_at'];
    }
    
    // Mettre à jour le statut en ligne
    public function updateOnlineStatus($status) {
        $query = "UPDATE " . $this->table_name . "
                  SET is_online = :status
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        $stmt->bindParam(":status", $status);
        $stmt->bindParam(":id", $this->id);
        
        if ($stmt->execute()) {
            return true;
        }
        return false;
    }
    
    // Mettre à jour la dernière connexion
    public function updateLastLogin() {
        $query = "UPDATE " . $this->table_name . "
                  SET last_login = NOW()
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":id", $this->id);
        
        if ($stmt->execute()) {
            return true;
        }
        return false;
    }
    
    // Mettre à jour le mot de passe par email
    public function updatePasswordByEmail($email, $new_password) {
        $query = "UPDATE " . $this->table_name . "
                  SET password = :password
                  WHERE email = :email";
        
        $stmt = $this->conn->prepare($query);
        
        $stmt->bindParam(":password", $new_password);
        $stmt->bindParam(":email", $email);
        
        if ($stmt->execute()) {
            return true;
        }
        return false;
    }
    
    // Récupérer la liste des contacts
    public function getContacts() {
        $query = "SELECT u.id, u.firstname, u.lastname, u.profile_image, u.is_online,
                         (SELECT m.message 
                          FROM messages m 
                          WHERE (m.sender_id = :user_id AND m.receiver_id = u.id) 
                             OR (m.sender_id = u.id AND m.receiver_id = :user_id) 
                          ORDER BY m.created_at DESC 
                          LIMIT 1) as last_message,
                         (SELECT m.created_at 
                          FROM messages m 
                          WHERE (m.sender_id = :user_id AND m.receiver_id = u.id) 
                             OR (m.sender_id = u.id AND m.receiver_id = :user_id) 
                          ORDER BY m.created_at DESC 
                          LIMIT 1) as last_message_time
                  FROM " . $this->table_name . " u
                  WHERE u.id != :user_id
                  ORDER BY last_message_time DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":user_id", $this->id);
        $stmt->execute();
        
        $contacts = array();
        
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $contact = array(
                "id" => $row['id'],
                "firstname" => $row['firstname'],
                "lastname" => $row['lastname'],
                "profile_image" => $row['profile_image'],
                "is_online" => $row['is_online'],
                "last_message" => $row['last_message'] ? $row['last_message'] : "Aucun message",
                "last_message_time" => $row['last_message_time'] ? date("H:i", strtotime($row['last_message_time'])) : ""
            );
            
            $contacts[] = $contact;
        }
        
        return $contacts;
    }
    
    // Mettre à jour le profil
    public function update() {
        $query = "UPDATE " . $this->table_name . "
                  SET firstname = :firstname,
                      lastname = :lastname,
                      email = :email";
        
        if (!empty($this->password)) {
            $query .= ", password = :password";
        }
        
        if (!empty($this->profile_image)) {
            $query .= ", profile_image = :profile_image";
        }
        
        $query .= " WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        
        // Nettoyer les données
        $this->firstname = htmlspecialchars(strip_tags($this->firstname));
        $this->lastname = htmlspecialchars(strip_tags($this->lastname));
        $this->email = htmlspecialchars(strip_tags($this->email));
        
        // Lier les valeurs
        $stmt->bindParam(":firstname", $this->firstname);
        $stmt->bindParam(":lastname", $this->lastname);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":id", $this->id);
        
        if (!empty($this->password)) {
            $stmt->bindParam(":password", $this->password);
        }
        
        if (!empty($this->profile_image)) {
            $stmt->bindParam(":profile_image", $this->profile_image);
        }
        
        if ($stmt->execute()) {
            return true;
        }
        return false;
    }
    
    // Vérifier le mot de passe actuel
    public function verifyPassword($password) {
        $query = "SELECT password FROM " . $this->table_name . " WHERE id = ? LIMIT 0,1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(1, $this->id);
        $stmt->execute();
        
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        return password_verify($password, $row['password']);
    }
}
?>