<?php
class User {
    // Connexion à la base de données et nom de la table
    private $conn;
    private $table_name = "users";
    
    // Propriétés de l'objet
    public $id;
    public $firstname;
    public $lastname;
    public $email;
    public $password;
    public $profile_image;
    public $is_online;
    public $last_login;
    public $created_at;
    
    // Constructeur avec $db comme connexion à la base de données
    public function __construct($db) {
        $this->conn = $db;
    }
    
    // Créer un nouvel utilisateur
    public function create() {
        // Requête d'insertion
        $query = "INSERT INTO " . $this->table_name . "
                  SET firstname = :firstname,
                      lastname = :lastname,
                      email = :email,
                      password = :password,
                      is_online = 0,
                      created_at = NOW()";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Nettoyer les données
        $this->firstname = htmlspecialchars(strip_tags($this->firstname));
        $this->lastname = htmlspecialchars(strip_tags($this->lastname));
        $this->email = htmlspecialchars(strip_tags($this->email));
        
        // Lier les valeurs
        $stmt->bindParam(":firstname", $this->firstname);
        $stmt->bindParam(":lastname", $this->lastname);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":password", $this->password);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Vérifier si l'email existe déjà
    public function emailExists($email) {
        // Requête pour vérifier si l'email existe
        $query = "SELECT id, firstname, lastname, password
                  FROM " . $this->table_name . "
                  WHERE email = ?
                  LIMIT 0,1";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier la valeur de l'email
        $stmt->bindParam(1, $email);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Vérifier si l'email existe
        if ($stmt->rowCount() > 0) {
            return true;
        }
        
        return false;
    }
    
    // Récupérer les informations d'un utilisateur par email
    public function getUserByEmail($email) {
        // Requête pour récupérer les informations de l'utilisateur
        $query = "SELECT id, firstname, lastname, email, profile_image
                  FROM " . $this->table_name . "
                  WHERE email = ?
                  LIMIT 0,1";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier la valeur de l'email
        $stmt->bindParam(1, $email);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Vérifier si l'email existe
        if ($stmt->rowCount() > 0) {
            return $stmt->fetch(PDO::FETCH_ASSOC);
        }
        
        return false;
    }
    
    // Connexion de l'utilisateur
    public function login($email, $password) {
        // Requête pour vérifier si l'email existe
        $query = "SELECT id, firstname, lastname, password
                  FROM " . $this->table_name . "
                  WHERE email = ?
                  LIMIT 0,1";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier la valeur de l'email
        $stmt->bindParam(1, $email);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Vérifier si l'email existe
        if ($stmt->rowCount() > 0) {
            // Récupérer les données de l'utilisateur
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // Vérifier le mot de passe
            if (password_verify($password, $row['password'])) {
                // Définir les valeurs des propriétés de l'objet
                $this->id = $row['id'];
                $this->firstname = $row['firstname'];
                $this->lastname = $row['lastname'];
                
                return true;
            }
        }
        
        return false;
    }
    
    // Lire les informations d'un utilisateur
    public function readOne() {
        // Requête pour lire un seul enregistrement
        $query = "SELECT u.id, u.firstname, u.lastname, u.email, u.profile_image, u.is_online, u.last_login, u.created_at
                  FROM " . $this->table_name . " u
                  WHERE u.id = ?
                  LIMIT 0,1";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier l'ID
        $stmt->bindParam(1, $this->id);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Récupérer la ligne
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Définir les valeurs des propriétés
        $this->firstname = $row['firstname'];
        $this->lastname = $row['lastname'];
        $this->email = $row['email'];
        $this->profile_image = $row['profile_image'];
        $this->is_online = $row['is_online'];
        $this->last_login = $row['last_login'];
        $this->created_at = $row['created_at'];
    }
    
    // Mettre à jour le statut en ligne
    public function updateOnlineStatus($status) {
        // Requête de mise à jour
        $query = "UPDATE " . $this->table_name . "
                  SET is_online = :status
                  WHERE id = :id";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":status", $status);
        $stmt->bindParam(":id", $this->id);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Mettre à jour la dernière connexion
    public function updateLastLogin() {
        // Requête de mise à jour
        $query = "UPDATE " . $this->table_name . "
                  SET last_login = NOW()
                  WHERE id = :id";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier l'ID
        $stmt->bindParam(":id", $this->id);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Récupérer la liste des contacts
    public function getContacts() {
        // Requête pour récupérer les contacts
        $query = "SELECT u.id, u.firstname, u.lastname, u.profile_image, u.is_online,
                         (SELECT m.message FROM messages m 
                          WHERE (m.sender_id = :user_id AND m.receiver_id = u.id) 
                             OR (m.sender_id = u.id AND m.receiver_id = :user_id) 
                          ORDER BY m.created_at DESC LIMIT 1) as last_message,
                         (SELECT m.created_at FROM messages m 
                          WHERE (m.sender_id = :user_id AND m.receiver_id = u.id) 
                             OR (m.sender_id = u.id AND m.receiver_id = :user_id) 
                          ORDER BY m.created_at DESC LIMIT 1) as last_message_time
                  FROM " . $this->table_name . " u
                  WHERE u.id != :user_id
                  ORDER BY last_message_time DESC";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier l'ID de l'utilisateur
        $stmt->bindParam(":user_id", $this->id);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Tableau de contacts
        $contacts = array();
        
        // Récupérer les résultats
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $contact = array(
                "id" => $row['id'],
                "firstname" => $row['firstname'],
                "lastname" => $row['lastname'],
                "profile_image" => $row['profile_image'],
                "is_online" => $row['is_online'],
                "last_message" => $row['last_message'] ? $row['last_message'] : "Aucun message",
                "last_message_time" => $row['last_message_time'] ? date("H:i", strtotime($row['last_message_time'])) : ""
            );
            
            $contacts[] = $contact;
        }
        
        return $contacts;
    }
    
    // Mettre à jour le profil de l'utilisateur
    public function update() {
        // Requête de mise à jour
        $query = "UPDATE " . $this->table_name . "
                  SET firstname = :firstname,
                      lastname = :lastname,
                      email = :email";
        
        // Ajouter le mot de passe à la requête s'il est défini
        if (!empty($this->password)) {
            $query .= ", password = :password";
        }
        
        // Ajouter l'image de profil à la requête si elle est définie
        if (!empty($this->profile_image)) {
            $query .= ", profile_image = :profile_image";
        }
        
        $query .= " WHERE id = :id";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Nettoyer les données
        $this->firstname = htmlspecialchars(strip_tags($this->firstname));
        $this->lastname = htmlspecialchars(strip_tags($this->lastname));
        $this->email = htmlspecialchars(strip_tags($this->email));
        
        // Lier les valeurs
        $stmt->bindParam(":firstname", $this->firstname);
        $stmt->bindParam(":lastname", $this->lastname);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":id", $this->id);
        
        // Lier le mot de passe s'il est défini
        if (!empty($this->password)) {
            $stmt->bindParam(":password", $this->password);
        }
        
        // Lier l'image de profil si elle est définie
        if (!empty($this->profile_image)) {
            $stmt->bindParam(":profile_image", $this->profile_image);
        }
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Vérifier le mot de passe actuel
    public function verifyPassword($password) {
        // Requête pour récupérer le mot de passe actuel
        $query = "SELECT password FROM " . $this->table_name . " WHERE id = ? LIMIT 0,1";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier l'ID
        $stmt->bindParam(1, $this->id);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Récupérer le mot de passe
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Vérifier le mot de passe
        return password_verify($password, $row['password']);
    }
    
    // Définir un token de réinitialisation de mot de passe
    public function setResetToken($email, $token) {
        // Requête de mise à jour
        $query = "UPDATE " . $this->table_name . "
                  SET reset_token = :token,
                      reset_token_expires = DATE_ADD(NOW(), INTERVAL 1 HOUR)
                  WHERE email = :email";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":token", $token);
        $stmt->bindParam(":email", $email);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Vérifier si un token de réinitialisation est valide
    public function verifyResetToken($token) {
        // Requête pour vérifier le token
        $query = "SELECT id FROM " . $this->table_name . "
                  WHERE reset_token = :token
                  AND reset_token_expires > NOW()
                  LIMIT 0,1";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier le token
        $stmt->bindParam(":token", $token);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Vérifier si le token existe et n'est pas expiré
        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $this->id = $row['id'];
            return true;
        }
        
        return false;
    }
    
    // Réinitialiser le mot de passe
    public function resetPassword($token, $new_password) {
        // Requête de mise à jour
        $query = "UPDATE " . $this->table_name . "
                  SET password = :password,
                      reset_token = NULL,
                      reset_token_expires = NULL
                  WHERE reset_token = :token";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":password", $new_password);
        $stmt->bindParam(":token", $token);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Mettre à jour le mot de passe
    public function updatePassword($new_password) {
        // Requête de mise à jour
        $query = "UPDATE " . $this->table_name . "
                  SET password = :password
                  WHERE id = :id";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":password", $new_password);
        $stmt->bindParam(":id", $this->id);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Récupérer les préférences de l'utilisateur
    public function getPreferences() {
        // Requête pour récupérer les préférences
        $query = "SELECT 
                    COALESCE(theme, 'light') as theme,
                    COALESCE(notifications, 1) as notifications,
                    COALESCE(sounds, 1) as sounds,
                    COALESCE(show_status, 1) as show_status,
                    COALESCE(read_receipts, 1) as read_receipts,
                    COALESCE(typing_indicator, 1) as typing_indicator
                  FROM user_preferences
                  WHERE user_id = ?";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier l'ID
        $stmt->bindParam(1, $this->id);
        
        // Exécuter la requête
        $stmt->execute();
        
        // Vérifier si des préférences existent
        if ($stmt->rowCount() > 0) {
            // Récupérer les préférences
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } else {
            // Créer des préférences par défaut
            $this->createDefaultPreferences();
            
            // Retourner les préférences par défaut
            return [
                'theme' => 'light',
                'notifications' => 1,
                'sounds' => 1,
                'show_status' => 1,
                'read_receipts' => 1,
                'typing_indicator' => 1
            ];
        }
    }
    
    // Créer des préférences par défaut
    private function createDefaultPreferences() {
        // Requête d'insertion
        $query = "INSERT INTO user_preferences
                  SET user_id = :user_id,
                      theme = 'light',
                      notifications = 1,
                      sounds = 1,
                      show_status = 1,
                      read_receipts = 1,
                      typing_indicator = 1";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier l'ID
        $stmt->bindParam(":user_id", $this->id);
        
        // Exécuter la requête
        $stmt->execute();
    }
    
    // Mettre à jour les préférences générales
    public function updatePreferences($theme, $notifications, $sounds) {
        // Requête de mise à jour
        $query = "INSERT INTO user_preferences
                  (user_id, theme, notifications, sounds)
                  VALUES (:user_id, :theme, :notifications, :sounds)
                  ON DUPLICATE KEY UPDATE
                  theme = :theme,
                  notifications = :notifications,
                  sounds = :sounds";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":user_id", $this->id);
        $stmt->bindParam(":theme", $theme);
        $stmt->bindParam(":notifications", $notifications);
        $stmt->bindParam(":sounds", $sounds);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
    
    // Mettre à jour les paramètres de confidentialité
    public function updatePrivacySettings($show_status, $read_receipts, $typing_indicator) {
        // Requête de mise à jour
        $query = "INSERT INTO user_preferences
                  (user_id, show_status, read_receipts, typing_indicator)
                  VALUES (:user_id, :show_status, :read_receipts, :typing_indicator)
                  ON DUPLICATE KEY UPDATE
                  show_status = :show_status,
                  read_receipts = :read_receipts,
                  typing_indicator = :typing_indicator";
        
        // Préparer la requête
        $stmt = $this->conn->prepare($query);
        
        // Lier les valeurs
        $stmt->bindParam(":user_id", $this->id);
        $stmt->bindParam(":show_status", $show_status);
        $stmt->bindParam(":read_receipts", $read_receipts);
        $stmt->bindParam(":typing_indicator", $typing_indicator);
        
        // Exécuter la requête
        if ($stmt->execute()) {
            return true;
        }
        
        return false;
    }
}
?>